package com.swvoyages.reservation;

import java.util.Random;

import electric.uddi.Bindings;
import electric.uddi.BusinessInfos;
import electric.uddi.IUDDI;
import electric.uddi.ServiceInfos;
import electric.uddi.TModel;
import electric.uddi.TModelInfos;
import electric.uddi.client.UDDIClient;

/*
* La classe <code>ReservationFinder</code> implmente l'interface 
* <code>ReservationLookup</code>. Elle est utilise pour dterminer la
* manire de rechercher l'une des implmentations du service type de
* rservation dont le nom est pass en paramtre, dans le registre UDDI
* dont l'URL de l'API de recherche est galement fournie en paramtre.
*
* @version  1.0
*/
public class ReservationFinder implements ReservationLookup {

	private static ReservationFinder uniqueInstance = new ReservationFinder();
	private static Random randomizer = new Random();

	/*
	* Construit une instance de la classe <code>ReservationFinder</code>.
	*/
	private ReservationFinder() {
	}

	/*
 	* Renvoie le singleton de la classe <code>ReservationFinder</code>.
 	*
 	* @return		ReservationFinder le singleton de la classe 
 	*				<code>ReservationFinder</code>.
	*/
	public static synchronized ReservationFinder getUniqueInstance() {
		if (uniqueInstance == null) {
			uniqueInstance = new ReservationFinder();
		}
		return uniqueInstance;
	}

	/*
	* Renvoie le tableau des coordonnes de rservation du
	* partenaire trouv. Celui-ci doit implmenter le service
	* type dont le nom est pass en paramtre et ses coordonnes
	* sont recherches dans le registre UDDI dont l'URL de l'API 
	* de recherche est communique.
	*
	* @param		inquireURL URL de l'API de recherche du
	*				registre UDDI slectionn
	* @param		tModelName nom du service type dont une
	*				implmentation est recherche
	*
	* @return		String[0] URL du fichier WSDL de description
	*				du service type implment
	* 				String[1] URL du point d'accs au service du
	*				partenaire
	*
	* @throws		ReservationException lance si une anomalie se
	*				produit durant la recherche d'un partenaire
	*/
	public synchronized String [] lookup(String inquireURL, String tModelName) 
		throws ReservationException {

    	try {
    		// Cration d'un client UDDI.
    		IUDDI uddi = new UDDIClient(inquireURL);

			// Recherche des services type dont le nom est spcifi.
    		TModelInfos tModelInfos = uddi.findTModels(tModelName, null);
    		if (tModelInfos.list.length==0) {
    			String message = "pas de service(s) type " + tModelName + " trouv(s)";
 				throw new ReservationException(message);
    		}

			// Slectionne au hasard l'un des services type trouvs. 
			int itmi = randomizer.nextInt(tModelInfos.list.length);
    		String tModelKey = tModelInfos.list[itmi].getTModelKey();

			// Recherche des entits mtier qui implmentent le service type.
    		String[] tModelKeys = new String[]{tModelKey};
    		BusinessInfos businessInfos = uddi.findBusinessesByTModelKeys(tModelKeys, null);
    		if (businessInfos.list.length==0) {
    			String message = "aucune entit mtier n'implmente le service type " + tModelKey;
 				throw new ReservationException(message);
    		}
    
			// Slectionne au hasard l'une des entits mtier trouves. 
			int ibi = randomizer.nextInt(businessInfos.list.length);
			String businessKey = businessInfos.list[ibi].getBusinessKey();

			// Recherche des services mtier qui implmentent le service type.
    		ServiceInfos serviceInfos = uddi.findServicesByTModelKeys(businessKey, tModelKeys, null);
    		if (serviceInfos.list.length==0) {
    			String message = "aucun service mtier n'implmente le service type " + tModelKey;
 				throw new ReservationException(message);
    		}

			// Slectionne au hasard l'un des services mtier trouvs. 
			int isi = randomizer.nextInt(serviceInfos.list.length);
			String serviceKey = serviceInfos.list[ibi].getServiceKey();


			// Recherche des liaisons qui implmentent le service type.
    		Bindings bindings = uddi.findBindings(serviceKey, tModelKeys, null);
    		if (bindings.list.length==0) {
    			String message = "aucune liaison n'implmente le service type " + tModelKey;
 				throw new ReservationException(message);
    		}

			// Slectionne au hasard l'une des liaisons trouves. 
			int ibt = randomizer.nextInt(bindings.list.length);

			// Rcupre l'URL du point d'accs au service slectionn.
			String address = bindings.list[ibt].getAccessPoint().getAddress();

			// Rcupre l'URL du modle WSDL du service slectionn.
    		TModel tModel = uddi.getTModel(tModelKey);
    		String wsdlURL = tModel.getOverview().getOverviewURL();
			return new String[]{wsdlURL, address};
		}
		catch (Throwable t) {
			t.printStackTrace();
			StringBuffer sb = new StringBuffer();
			sb.append("une ");
			sb.append(t.getClass());
			sb.append(" s'est produite durant la recherche d'un partenaire de rservation");
			if ((t.getMessage()!=null) && (!(t.getMessage().equals("")))) {
				sb.append(" : ");
				sb.append(t.getMessage());
			}
			throw new ReservationException(sb.toString(), t);
		}
    }
}
